unit FBReplService;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, SvcMgr, Dialogs,
  ExtCtrls, IB_Events, DB, IBODataset, IB_Components, IBAPI, IBDBAPI,
  Rascomp32, IniFiles, IdMessage, IdBaseComponent, IdComponent, Reg, FBReplConst,
  IdTCPConnection, IdTCPClient, IdMessageClient, IdSMTP,
  IdExplicitTLSClientServerBase, IdSMTPBase;

type
  ParamRec = record
       ParamPtr    : pointer;
       NullPtr     : pointer;
       BlobSizePtr : PSDWORD;
       DataType    : SWORD;
  end;
  StmtPtr = ^StmtPtrNode;
  LocsPtr = ^LocsPtrNode;
  StmtPtrNode = record
    TableName  : array[0..32] of Char;
    Operation  : array[0..1] of Char;
    SqlStmt    : Pointer;
    Link  : StmtPtr;
  end;
  LocsPtrNode = record
    LocPath  : Pointer;
    RService : array[0..49] of Char;
    RUser    : array[0..49] of Char;
    RPass    : array[0..49] of Char;
    UserName : array[0..49] of Char;
    Password : array[0..49] of Char;
    IdLoc : Integer; { The actual Location Id }
    Link  : LocsPtr;
  end;
  HeadPtr = ^StmtPtr;
  LocsHeadPtr = ^LocsPtr;

  TFBReplicator = class(TService)
    ReplDB: TIBODatabase;
    DBSource: TIBODatabase;
    EvSourceDB: TIBODatabase;
    QChanges: TIBOQuery;
    QChangesCOUNT: TIntegerField;
    QueryStmt: TIBOQuery;
    QueryStmtTABLENAME: TStringField;
    QueryStmtOPTYPE: TStringField;
    QueryStmtSQLSTMT: TStringField;
    QueryStmtMORE: TSmallintField;
    QueryLocs: TIBOQuery;
    QueryLocsLOC_ID: TIntegerField;
    QueryLocsLOC_PATH: TStringField;
    QueryLocsRAS_SERVICENAME: TStringField;
    QueryLocsRAS_USER: TStringField;
    QueryLocsRAS_PASSWORD: TStringField;
    QueryLocsUSERNAME: TStringField;
    QueryLocsPASSWD: TStringField;
    QLocChanges: TIBOQuery;
    QLocChangesLOC_ID: TIntegerField;
    QSrcLocs: TIBOQuery;
    QSrcLocsSOURCE_PATH: TStringField;
    QSrcLocsUSERNAME: TStringField;
    QSrcLocsPASSWD: TStringField;
    QSrcLocsSOURCE_SERVER: TStringField;
    IBEvent: TIB_Events;
    TDoReplication: TTimer;
    TReplInterval: TTimer;
    IdSMTP1: TIdSMTP;
    IdMessage1: TIdMessage;
    QSrcLocsID: TIntegerField;
    QSrcLocsREPLUSER: TStringField;
    QSrcLocsREPLPASSWD: TStringField;
    procedure ServiceStart(Sender: TService; var Started: Boolean);
    procedure TDoReplicationTimer(Sender: TObject);
    procedure TReplIntervalTimer(Sender: TObject);
    procedure ServiceStop(Sender: TService; var Stopped: Boolean);
  private
    procedure AddLocsLL(Head: LocsHeadPtr; LocPath, RasServ, RasUser,
      RasPass: PChar; IdLoc: Integer; User, Passwd: PChar);
    procedure AddStmtLL(Head: HeadPtr; TableName, Operation,
      SqlStr: PChar);
    function CacheSqlStmts: Integer;
    procedure CloseChanges;
    function ConnectDb(LocId: Integer; LocPtr: LocsPtr): Integer;
    function DeleteChangeRec: RETCODE;
    procedure DestroyLocsLL(Head: LocsHeadPtr);
    procedure DestroyStmtLL(Head: HeadPtr);
    procedure DisconnectDb(IdLoc: Integer);
    procedure FreeSqlStmts;
    function GetChanges: RETCODE;
    procedure HandleError(ErrorMsg: string; Error: Boolean);
    procedure IBEvent1EventAlert(Sender: TObject; EventName: string;
      EventCount: Integer; var CancelAlerts: Boolean);
    function OpenChanges: RETCODE;
    procedure ReplicateData;
    function SearchLocsLL(Head: LocsPtr; IdLoc: Integer): LocsPtr;
    function SearchStmtLL(Head: StmtPtr; TableName,
      Operation: PChar): PChar;
    procedure SetParams;
    function SyncDelete: RETCODE;
    function SyncInsUpd: RETCODE;
    procedure SyncSrcAndTarget;
    function CreateLogFileName: String;
    procedure SendStartingEmail;
    procedure SendStoppingEmail;
    { Private declarations }
  public
//    ErrorFile : array[0..254] of Char;
    function GetServiceController: TServiceController; override;
    { Public declarations }
  end;

var
  FBReplicator: TFBReplicator;
  StmtHead : StmtPtr;
  LocsHead : LocsPtr;
  CachedEm : Boolean;
  TotalLocs : Integer;
  SrcHandle,TargetHandle : HDBC;
  ChgStmt,ChgStmt2,SrcStmt,TargetStmt : HSTMT;
  ChgCode,ChgLocId, tempLocID  : Integer;
  ChgTableKey : Int64;
  ChgTableName : array[0..33] of Char;
  ChgOperation : array[0..1] of Char ;
  SomeNull1,SomeNull2,SomeNull3,SomeNull4,SomeNull5 : Boolean;
  SqlStmtPtr : PChar;
  Running,Reg4Event,FatalError : Boolean;

implementation

{$R *.DFM}

function TFBReplicator.CreateLogFileName: String;
var
  CurrentTimeStamp: TDateTime;
  Year, Month, Day: Word;
begin
  CurrentTimeStamp := Now;
  DecodeDate(CurrentTimeStamp, Year, Month, Day);
  Result := IntToStr(Year) +'-'+ IntToStr(Month) +'-'+ IntToStr(Day) +'-';
end;

function TFBReplicator.CacheSqlStmts: Integer;
var
  SqlStmts:Integer;
  TableName : array[0..32] of Char;
  Operation : array[0..1] of Char;
  SqlStr : array[0..2047] of Char;
  TmpStr, LocPath : string;
  IdLoc : Integer;
begin
  if ( not CachedEm) then begin
    SqlStmts := 0;
    SqlStr := '';
    if (ReplDB.InTransaction) then
      ReplDB.Commit;
    ReplDB.StartTransaction;
    QueryStmt.Open;
    QueryStmt.First;
    StrCopy(TableName,PChar(QueryStmtTABLENAME.value));
    StrCopy(Operation,PChar(QueryStmtOPTYPE.value));
    while( QueryStmt.EOF <> True ) do  begin
      if (QueryStmtOPTYPE.value <> Operation) or (QueryStmtTABLENAME.value <> TableName) then begin
        Inc(SqlStmts);
        AddStmtLL(@(StmtHead),TableName,Operation,SqlStr);
        StrCopy(TableName,PChar(QueryStmtTABLENAME.value));
        StrCopy(Operation,PChar(QueryStmtOPTYPE.value));
        SqlStr := '';
      end;
      TmpStr := QueryStmtSQLSTMT.value;
      StrLCat(SqlStr,PChar(TmpStr),Sizeof(SqlStr)-1);
      QueryStmt.Next;
    end;
    if (TableName <> '') then
         AddStmtLL(@(StmtHead),TableName,Operation,SqlStr);
    QueryStmt.Close;
    { See how many locations we have to deal with and cache all the dbpaths/locid while we're at it }
    QueryLocs.Open;
    QueryLocs.First;
    TotalLocs :=0;
    while( QueryLocs.EOF <> True ) do begin
      Inc(TotalLocs);
      AddLocsLL(@(LocsHead),PChar(QueryLocsLOC_PATH.value),PChar(QueryLocsRAS_SERVICENAME.value),PChar(QueryLocsRAS_USER.value),PChar(QueryLocsRAS_PASSWORD.value),QueryLocsLOC_ID.value,PChar(QueryLocsUSERNAME.value),PChar(QueryLocsPASSWD.value));
      QueryLocs.Next;
    end;
    ReplDB.Commit;
    CachedEm := True;
  end;
  CacheSqlStmts := TotalLocs;
end;

procedure TFBReplicator.FreeSqlStmts;
begin
  DestroyStmtLL(@(StmtHead));
  DestroyLocsLL(@(LocsHead));
end;

procedure TFBReplicator.SetParams;
var
  Len : Integer;
//  TmpError : array[0..254] of Char;
  SomeNum,n1,PathLen: Integer;
begin
  if (StrComp('Not Found',PChar(srReplPath)) = 0) or(StrComp('Not Found',PChar(srReplServer))=0) then begin
//    ShowMessage('Cannot find registry settings or the ReplMgmt section is missing.  See readme.txt included with Replication Server zip file.');
    TReplInterval.Enabled := False;
    HandleError('srReplPath not found or srReplServer no found', True);
    //FatalError := True;
    //Application.Terminate;
  end else begin
//    StrCopy(ErrorFile,PChar(srReplLogFile));
    if (not ReplDB.Connected) then begin
      ReplDB.Server := srReplServer;
      ReplDB.Path := srReplPath;
      ReplDB.Username := srReplUser;
      ReplDB.Password := srReplPWord;
      ReplDB.Open;
    end;
    if ( not ReplDB.Connected) then
      FatalError := True;
  end;
end;

procedure TFBReplicator.HandleError(ErrorMsg:string; Error: Boolean);
var
  Len : Integer;
  F:Textfile;
  FileId : Integer;
  CleanString, slFileName: String;
begin
  slFileName := CreateLogFileName();
  slFileName := ExcludeTrailingPathDelimiter(ExtractFilePath(srReplLogFile))+'\'+slFileName+ExtractFileName(srReplLogFile);
  if (not FileExists(slFileName)) then begin
    FileId := FileCreate(slFileName);
    if FileId > 0 then begin
      FileClose(FileId);
    end;
  end;
  AssignFile(F,slFileName);
  Append(F);
  CleanString := StringReplace(ErrorMsg, '|', ' ', [rfReplaceAll]);
  Writeln(F,DateTimeToStr(Now) + ' '+CleanString);
  CloseFile(F);
  if Error then Inc(AnyErrors);
  if Error then Inc(TotalErrors);
  if Error then LatestError := CleanString;
end;

procedure TFBReplicator.DestroyStmtLL(Head:HeadPtr);
var
  CurPtr,NextPtr : StmtPtr;
begin
  CurPtr := Head^;
  if ( CurPtr <> nil) then begin
    while (CurPtr^.link <> nil) do begin
      NextPtr := CurPtr^.link;
      if CurPtr^.SqlStmt <> nil then
        FreeMem(PChar(CurPtr^.SqlStmt));
      FreeMem(CurPtr);
      CurPtr := NextPtr;
    end;
    FreeMem(CurPtr^.SqlStmt);
    FreeMem(CurPtr);
    Head^ := nil;
  end;
end;

procedure TFBReplicator.AddStmtLL(Head:HeadPtr;TableName,Operation,SqlStr:PChar);
var
  LenStr : Integer;
  TmpPtr,NewPtr : StmtPtr;
begin
  New(NewPtr);
  StrCopy(NewPtr^.Tablename,TableName);
  StrCopy(NewPtr^.Operation,Operation);
  LenStr := StrLen(SqlStr);
  NewPtr^.SqlStmt := AllocMem(LenStr+1);
  StrCopy(NewPtr^.SqlStmt,SqlStr);
  NewPtr^.link := nil;
  TmpPtr := Head^;
  if (TmpPtr = nil) then
    Head^ := NewPtr   {First one on the list}
  else begin
    while (TmpPtr^.link <> nil) do   { stick it on the end }
      TmpPtr := TmpPtr^.link;
    TmpPtr^.link := NewPtr;
  end;
end;

function  TFBReplicator.SearchStmtLL(Head:StmtPtr;TableName,Operation:PChar): PChar;
var
  CurPtr : StmtPtr;
  Foundit : Integer;
begin
  CurPtr := Head;
  Foundit := 0;
  if ( CurPtr <> nil) then begin
    while (CurPtr <> nil) and (Foundit = 0) do begin
      if ( StrIComp(CurPtr^.TableName,TableName)=0) and (StrIComp(CurPtr^.Operation,Operation)=0) then
        Foundit := 1
      else
        CurPtr := CurPtr^.link;
    end;
  end;
  if CurPtr = nil then
    SearchStmtLL := nil
  else
    SearchStmtLL := CurPtr^.SqlStmt;
end;

procedure TFBReplicator.DestroyLocsLL(Head:LocsHeadPtr);
var
  CurPtr,NextPtr : LocsPtr;
begin
  CurPtr := Head^;
  if ( CurPtr <> nil) then begin
    while (CurPtr^.link <> nil) do begin
      NextPtr := CurPtr^.link;
      if CurPtr^.LocPath <> nil then
        FreeMem(PChar(CurPtr^.LocPath));
      FreeMem(CurPtr);
      CurPtr := NextPtr;
    end;
    FreeMem(CurPtr^.LocPath);
    FreeMem(CurPtr);
    Head^ := nil;
  end
end;

procedure TFBReplicator.AddLocsLL(Head:LocsHeadPtr;LocPath,RasServ,RasUser,RasPass:PChar;IdLoc :Integer;User,Passwd:PChar);
var
  LenStr : Integer;
  TmpPtr,NewPtr : LocsPtr;
begin
  New(NewPtr);
  NewPtr^.IdLoc := IdLoc;
  LenStr := StrLen(LocPath);
  NewPtr^.LocPath := AllocMem(LenStr+1);
  StrCopy(NewPtr^.LocPath,LocPath);
  StrCopy(NewPtr^.RService, RasServ);
  StrCopy(NewPtr^.RUser, RasUser);
  StrCopy(NewPtr^.RPass, RasPass);
  StrCopy(NewPtr^.UserName, User);
  StrCopy(NewPtr^.Password, Passwd);
  NewPtr^.link := nil;
  TmpPtr := Head^;
  if (TmpPtr = nil) then
    Head^ := NewPtr   {First one on the list}
  else begin
    while (TmpPtr^.link <> nil) do   { stick it on the end }
      TmpPtr := TmpPtr^.link;
    TmpPtr^.link := NewPtr;
  end;
end;

function  TFBReplicator.SearchLocsLL(Head:LocsPtr;IdLoc:Integer): LocsPtr;
var
  CurPtr : LocsPtr;
  Foundit : Integer;
begin
  CurPtr := Head;
  Foundit := 0;
  if ( CurPtr <> nil) then begin
    while (CurPtr <> nil) and (Foundit = 0) do begin
      if ( IdLoc =  CurPtr^.IdLoc ) then
        Foundit := 1
      else
        CurPtr := CurPtr^.link;
    end;
  end;
  if CurPtr = nil then
    SearchLocsLL := nil
  else
    SearchLocsLL := CurPtr;
end;

function TFBReplicator.ConnectDb(LocId:Integer;LocPtr:LocsPtr):Integer;
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
  DbName : string;
  DbStatus : Integer;
begin
   retval := SQL_SUCCESS;
   FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
   if (LocId = -1) then begin
     SrcHandle := nil;
     SQLAllocConnect(@SrcHandle);
     SQLSetConnectOption( SrcHandle,SQL_AUTOCOMMIT,SQL_AUTOCOMMIT_OFF);
     retval := SQLConnect(SrcHandle,PChar(SrcPathDB),PChar(SrcUsername),PChar(SrcPassword));
     if ( retval <> SQL_SUCCESS) then begin
       SQLError(SrcHandle,nil,ErrorBuf);
       HandleError('Src Connect Failed: '+ ErrorBuf, True);
     end;
   end else begin
     if (LocPtr <> nil) then begin
       TargetUser := LocPtr^.UserName;
       TargetPwd := LocPtr^.Password;
       TargetPath := StrPas(PChar(LocPtr^.LocPath));
       { Is there a RAS service affilliated with this location? }
(*       if (Length(LocPtr^.RService) > 0) then  begin
         dialer.REntryName := LocPtr^.RService;
         dialer.RUserName := LocPtr^.RUser;
         dialer.RPassword := LocPtr^.RPass;
         {Wait for the RAS connection to take place.}
         dialer.ShowModal;
         if (not dialer.IsConnected) then begin
           retval := SQL_ERROR;
           HandleError('Target '+IntToStr(LocId)+' Connect Failed: Could not connect to associated RAS Service('+Form2.REntryName+')');
         end
       end; *)
     end else begin
       retval := SQL_ERROR;
       HandleError('Could not find Target location linked-list info', True);
     end;
     if (retval = SQL_SUCCESS) then begin
       TargetHandle := nil;
       SQLAllocConnect(@TargetHandle);
       SQLSetConnectOption( TargetHandle,SQL_AUTOCOMMIT,SQL_AUTOCOMMIT_OFF);
       retval := SQLConnect(TargetHandle,PChar(TargetPath),PChar(TargetUser),PChar(TargetPwd));
       if ( retval <> SQL_SUCCESS) then begin
         SQLError(TargetHandle,nil,ErrorBuf);
         HandleError('Target '+IntToStr(LocId)+' Connect Failed: '+ ErrorBuf, True);
       end
     end else begin
       retval := SQL_ERROR;
       HandleError('Target '+IntToStr(LocId)+' Connect Failed: Could not connect to associated RAS Service', True);
     end;
   end;
   ConnectDb := retval;
end;

procedure TFBReplicator.DisconnectDb(IdLoc:Integer);
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
begin
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  if (IdLoc = -1) then begin
    retval := SQLDisconnect(@SrcHandle);
    if ( retval <> SQL_SUCCESS) then begin
      SQLError(SrcHandle,nil,ErrorBuf);
      HandleError('Source '+IntToStr(IdLoc)+' Disconnect Failed: '+ ErrorBuf, True);
    end;
    if ( DBSource.Connected) then
      DBSource.Connected := False;
  end else begin
    retval := SQLDisconnect(@TargetHandle);
    if ( retval <> SQL_SUCCESS) then begin
      SQLError(TargetHandle,nil,ErrorBuf);
      HandleError('Target '+IntToStr(IdLoc)+' Disconnect Failed: '+ ErrorBuf, True);
    end;
//    if (dialer.IsConnected) then begin
//      dialer.RAS1.Disconnect;
//      dialer.IsConnected := False;
//    end;
  end;
end;

function TFBReplicator.OpenChanges:RETCODE;
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
begin
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  retval := SQLAllocStmt( SrcHandle,@ChgStmt);
  if ( retval = SQL_SUCCESS) then begin
    retval := SQLPrepare( ChgStmt, PChar(ChangeSelect) , 0 );
    if brVerbose then HandleError('LOCID: '+IntToStr(tempLocId)+'; (OpenChanges)', False);
    if ( retval = SQL_SUCCESS) then begin
      SQLBindCol( ChgStmt,1, SQL_INT64+1,@ChgCode,@SomeNull1,nil);
      SQLBindCol( ChgStmt,2, SQL_VARYING+1,@ChgTableName,@SomeNull2,nil);
      SQLBindCol( ChgStmt,3, SQL_INT64+1,@ChgTableKey,@SomeNull3,nil);
      SQLBindCol( ChgStmt,4, SQL_TEXT+1,@ChgOperation,@SomeNull4,nil);
      SQLBindParameter( ChgStmt,1, SQL_LONG+1,@ChgLocId,@SomeNull5,nil,0);
      if brVerbose then HandleError('After Col Bind LOCID: '+IntToStr(tempLocId)+'; (OpenChanges)', False);
      retval := SQLExecute( ChgStmt );
      if ( retval <> SQL_SUCCESS) then begin
        SQLError(SrcHandle,ChgStmt,ErrorBuf);
        HandleError('ChgExecute Failed: for LocId='+IntToStr(tempLocId)+' SqlStmt='+ChangeSelect+' Error='+ ErrorBuf, True);
        SQLFreeStmt( ChgStmt,SQL_DROP);
      end;
    end else begin
      SQLError(SrcHandle,ChgStmt,ErrorBuf);
      HandleError('ChgPrepare Failed: for LocId='+IntToStr(tempLocId)+' SqlStmt='+ChangeSelect+' Error='+ErrorBuf, True);
      SQLFreeStmt( ChgStmt,SQL_DROP);
    end;
  end else begin
    SQLError(SrcHandle,nil,ErrorBuf);
    HandleError('Chg AllocFailed for LocId='+IntToStr(tempLocId)+' Error= '+ ErrorBuf, True);
  end;
  OpenChanges := retval;
end;

function TFBReplicator.DeleteChangeRec:RETCODE;
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
begin
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  retval := SQLAllocStmt( SrcHandle,@ChgStmt2);
  if ( retval = SQL_SUCCESS) then begin
    retval := SQLPrepare( ChgStmt2, PChar(ChangeDelete) , 0 );
    if ( retval = SQL_SUCCESS) then begin
      SQLBindParameter( ChgStmt2,1, SQL_INT64+1,@ChgCode,@SomeNull1,nil,0);
      retval := SQLExecute( ChgStmt2 );
      if ( retval <> SQL_SUCCESS) then begin
        SQLError(SrcHandle,ChgStmt2,ErrorBuf);
        HandleError('ChgDelete Failed LocId='+IntToStr(ChgCode)+' SqlStmt='+ChangeDelete+ 'Error='+ErrorBuf, True);
      end;
    end else begin
      SQLError(SrcHandle,ChgStmt2,ErrorBuf);
      HandleError('ChgDelPrepare Failed LocId='+IntToStr(ChgCode)+' SqlStmt='+ ChangeDelete+' Error='+ErrorBuf, True);
    end;
    SQLFreeStmt( ChgStmt2,SQL_DROP);
  end else begin
    SQLError(SrcHandle,nil,ErrorBuf);
    HandleError('ChgDel AllocFailed LocId='+IntToStr(tempLocId)+' Error='+ ErrorBuf, True);
  end;
  DeleteChangeRec := retval;
end;

function TFBReplicator.GetChanges:RETCODE;
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
  tempLoc: Integer;
begin
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  if brVerbose then HandleError('Fetching change statement; (GetChanges)', False);
  if brVerbose then HandleError('Before SQLFetch ChgLocID: '+IntToStr(tempLocId)+'; (GetChanges)', False);
  tempLoc := ChgLocID;
  retval := SQLFetch( ChgStmt );
  if brVerbose then HandleError('ChgLocID: '+IntToStr(tempLocId)+'; (GetChanges)', False);
//  ChgLocID := tempLoc;
  if (retval <> SQL_SUCCESS) then begin
     if (retval <> 100) then begin
       SQLError(SrcHandle,ChgStmt,ErrorBuf);
       HandleError('ChgFetch Failed LocId='+IntToStr(tempLocId)+' SqlStmt='+ ChangeSelect+ ' Error='+ErrorBuf, True);
     end;
    if brVerbose then HandleError('Returning retval: '+IntToStr(retval)+'; (GetChanges)', False);
  end else begin
    if brVerbose then HandleError('Returning retval: SQL_SUCCESS'+'; (GetChanges)', False);
  end;
  GetChanges := retval;
end;

procedure TFBReplicator.CloseChanges;
begin
  SQLFreeStmt( ChgStmt,SQL_DROP);
end;

function TFBReplicator.SyncDelete:RETCODE;
var
  retval : RETCODE;
  ErrorBuf : array[0..511] of Char;
begin
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  retval := SQLAllocStmt( TargetHandle,@TargetStmt);
  if ( retval = SQL_SUCCESS) then begin
    retval := SQLPrepare( TargetStmt, PChar(SqlStmtPtr) , 0 );
    if ( retval = SQL_SUCCESS) then begin
      SomeNull1 := False;
      SQLBindParameter( TargetStmt,1, SQL_INT64+1,@ChgTableKey,@SomeNull1,nil,0);
      retval := SQLExecute( TargetStmt );
      if ( retval <> SQL_SUCCESS) then begin
        SQLError(TargetHandle,TargetStmt,ErrorBuf);
        HandleError('TgtDelete Failed: LocId='+IntToStr(tempLocId)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SqlStmtPtr+' Error='+ErrorBuf, True);
      end;
    end else begin
      SQLError(TargetHandle,TargetStmt,ErrorBuf);
      HandleError('TgtDelPrepare Failed: LocId='+IntToStr(tempLocId)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SqlStmtPtr+' Error='+ErrorBuf, True);
    end;
    SQLFreeStmt( TargetStmt,SQL_DROP);
  end else begin
    SQLError(TargetHandle,nil,ErrorBuf);
    HandleError('TgtDel AllocFailed: LocId='+IntToStr(tempLocId)+' Error='+ ErrorBuf, True);
  end;
  SyncDelete := retval;
end;

function TFBReplicator.SyncInsUpd:RETCODE;
var
  retval1,retval2 : RETCODE;
  ErrorBuf : array[0..511] of Char;
  SrcSqlPtr : PChar;
  DataType,DataLen:SWORD;
  ParamNum,NumIn,NumOut,i : UWORD;
  InList,OutList : array[1..MaxColumns] of ParamRec;
  BoolPtr : ^Boolean;
  TmpChar : array[0..1] of Char;
begin
  retval1 := 0;
  retval2 := 0;
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  if brVerbose then HandleError('Working with LOCID: '+IntToStr(tempLocId)+';(SyncInsUpd)', False);
  retval1 := SQLAllocStmt( SrcHandle,@SrcStmt);
  retval2 := SQLAllocStmt( TargetHandle,@TargetStmt);
  if brVerbose then HandleError('Working with LOCID: '+IntToStr(tempLocId)+';(SyncInsUpd)', False);
  if ( retval1 = SQL_SUCCESS) and (retval2 = SQL_SUCCESS) then begin
    TmpChar := 'S';
    SrcSqlPtr := SearchStmtLL(StmtHead,ChgTableName,TmpChar);
    retval1 := SQLPrepare( SrcStmt, SrcSqlPtr , 0 );
    if ( retval1 = SQL_SUCCESS) then begin
      NumOut := 1;
      { For each column in a Select clause, we'll allocate space. SQLGetCol() will tell us
        the Datatype and Length.  Once we have the proper space allocated, we can "bind" the
        output data to the variable pointed to by the allocated space using SQLBindCol(). }
      retval1 := SQLGetCol( SrcStmt,NumOut,DataType,DataLen);
      while (retval1 <> 100) do begin
        OutList[NumOut].NullPtr := AllocMem(sizeof(Boolean));
        OutList[NumOut].BlobSizePtr := nil;
        BoolPtr := OutList[NumOut].NullPtr;
        BoolPtr^ := False;
        case DataType of
          SQL_DATE+1,SQL_DATE: begin
          { Dates are passed in/out as strings of the format mm/dd/yyyy hh:mm:ss SQL_TIMESTAMP}
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*20);
            SQLBindCol( SrcStmt,NumOut, SQL_DATE+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_DATE+1;
          end;
          SQL_TYPE_DATE+1,SQL_TYPE_DATE: begin
          { Dates are passed in/out as strings of the format mm/dd/yyyy ?}
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*20);
            SQLBindCol( SrcStmt,NumOut, SQL_TYPE_DATE+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_TYPE_DATE+1;
          end;
          SQL_TYPE_TIME+1,SQL_TYPE_TIME: begin
          { Dates are passed in/out as strings of the format hh:mm:ss }
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*20);
            SQLBindCol( SrcStmt,NumOut, SQL_TYPE_TIME+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_TYPE_TIME+1;
          end;
          SQL_TEXT+1,SQL_TEXT: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*DataLen+1);
            SQLBindCol( SrcStmt,NumOut, SQL_TEXT+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_TEXT+1;
          end;
          SQL_VARYING+1,SQL_VARYING: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*DataLen+1);
            SQLBindCol( SrcStmt,NumOut, SQL_VARYING+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_VARYING+1;
          end;
          SQL_INT64,SQL_INT64+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Int64));
            SQLBindCol( SrcStmt,NumOut, SQL_INT64+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_INT64+1;
          end;
          SQL_LONG,SQL_LONG+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(LongInt));
            SQLBindCol( SrcStmt,NumOut, SQL_LONG+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_LONG+1;
          end;
          SQL_SHORT,SQL_SHORT+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Short));
            SQLBindCol( SrcStmt,NumOut, SQL_SHORT+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_SHORT+1;
          end;
          SQL_FLOAT,SQL_FLOAT+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Double));
            SQLBindCol( SrcStmt,NumOut, SQL_FLOAT+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_FLOAT+1;
          end;
          SQL_DOUBLE,SQL_DOUBLE+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Double));
            SQLBindCol( SrcStmt,NumOut, SQL_DOUBLE+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,nil);
            OutList[NumOut].DataType := SQL_DOUBLE+1;
          end;
          SQL_BLOB,SQL_BLOB+1: begin
            OutList[NumOut].ParamPtr := AllocMem(sizeof(Char)*BlobMaxSize);
            OutList[NumOut].BlobSizePtr := AllocMem(sizeof(SDWORD));
            { OutList[NumOut].BlobSizePtr^ := DataLen;}
            OutList[NumOut].BlobSizePtr^ := sizeof(Char)*BlobMaxSize;
            { DataLen is filled in when we called SQLGetCol. DataLen contains the segment size for this blob
              field. After calling SQLFetch() this variable will contain the actual size of the Blob returned. }
            SQLBindCol( SrcStmt,NumOut, SQL_BLOB+1,OutList[NumOut].ParamPtr,OutList[NumOut].NullPtr,OutList[NumOut].BlobSizePtr);
            OutList[NumOut].DataType := SQL_BLOB+1;
          end;
        end;
        Inc(NumOut);
        retval1 := SQLGetCol( SrcStmt,NumOut,DataType,DataLen);
      end;
      SomeNull1 := False;
      SQLBindParameter( SrcStmt,1, SQL_INT64+1,@ChgTableKey,@SomeNull1,nil,0);
      retval2 := SQLPrepare( TargetStmt, PChar(SqlStmtPtr) , 0 );
      if (retval2 = SQL_SUCCESS) then begin
        { SQLBindCol is for output params SQLBindParameter is for input params}
        { The input params to the other dbs stmt(hstmt2) are the output params from the first dbs
          stmt(hstmt1). If this were not the case then, we'd obviously have to allocate space for the
          variables we were going to pass in.}
        i := 1;
        while i < NumOut do begin
          if (OutList[i].BlobSizePtr = nil) then
            SQLBindParameter( TargetStmt,i, OutList[i].DataType,OutList[i].ParamPtr,OutList[i].NullPtr,OutList[i].BlobSizePtr,0)
          else
            SQLBindParameter( TargetStmt,i, OutList[i].DataType,OutList[i].ParamPtr,OutList[i].NullPtr,OutList[i].BlobSizePtr,OutList[i].BlobSizePtr^);
            Inc(i);
        end;
        if (ChgOperation = 'U') then begin
          { If we're issuing an Update statement, we'll have one more parameter to Bind into the
            Input data section, i.e. The TableKey }
           SomeNull1 := False;
           SQLBindParameter( TargetStmt, NumOut, SQL_INT64+1, @ChgTableKey, @SomeNull1, nil, 0);
        end;
        retval1 := SQLExecute( SrcStmt );
        retval2 := SQLFetch( SrcStmt );
        while ( (retval1  = SQL_SUCCESS) and (retval2 = SQL_SUCCESS)) do begin
          retval2 := SQLExecute( TargetStmt );
          if (retval2 = SQL_SUCCESS) then
            retval1 := SQLFetch( SrcStmt );
        end;
        if ( (retval1  = 100) and (retval2 = SQL_SUCCESS)) then begin
            retval1 := SQL_SUCCESS;
        end else begin
          if (retval2 <> SQL_SUCCESS) then begin
            if (retval2 <> 100) then begin
              if (retval2 = -23) then begin
                HandleError('Not enough memory allocated for a Blob field.  Make sure BlobMaxBytes is set to a large enough value in repl.ini', True);
                retval1 := SQL_ERROR;
              end else begin
                SQLError(TargetHandle,TargetStmt,ErrorBuf);
                HandleError('InsUpd TgtEx Failed: LocId='+IntToStr(tempLocID)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SqlStmtPtr+' Error='+ErrorBuf, True);
                retval1 := SQL_ERROR;
              end;
            end;
          end else begin
            SQLError(SrcHandle,SrcStmt,ErrorBuf);
            HandleError('InsUpd SrcFetch Failed: LocId='+IntToStr(tempLocID)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SrcSqlPtr+' Error='+ErrorBuf, True);
            retval1 := SQL_ERROR;
          end;
        end;
      end else begin { Target Prepare Failed }
        SQLError(TargetHandle,TargetStmt,ErrorBuf);
        if brVerbose then HandleError('TgtPrep with LOCID: '+IntToStr(tempLocId)+';(SyncInsUpd)', False);
        HandleError('InsUpd TgtPrep Failed: LocId='+IntToStr(tempLocID)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SqlStmtPtr+' Error='+ErrorBuf, True);
        retval1 := SQL_ERROR;
      end;
    end else begin  { Source Prepare Failed }
      SQLError(SrcHandle,SrcStmt,ErrorBuf);
      HandleError('InsUpd SrcPrep Failed: LocId='+IntToStr(tempLocID)+' Key='+IntToStr(ChgTableKey)+ ' SqlStmt='+SrcSqlPtr+' Error='+ErrorBuf, True);
    end;
      SQLFreeStmt( SrcStmt,SQL_DROP);
      SQLFreeStmt( TargetStmt,SQL_DROP);
      i := 1;
      while i < NumOut do begin
        if OutList[i].DataType = SQL_BLOB+1 then
          FreeMem( OutList[i].BlobSizePtr);
        FreeMem( OutList[i].ParamPtr);
        FreeMem( OutList[i].NullPtr);
        Inc(i);
      end;
    end else begin
      if retval1 <> SQL_SUCCESS then begin
        SQLError(SrcHandle,nil,ErrorBuf);
        HandleError('InsUpd SrcAlloc Failed: LocId'+IntToStr(tempLocID)+' Error='+ ErrorBuf, True);
      end else begin
        SQLError(TargetHandle,nil,ErrorBuf);
        HandleError('InsUpd Target Alloc Failed: LocId'+IntToStr(tempLocID)+' Error='+ ErrorBuf, True);
        retval1 := SQL_ERROR;
      end;
    end;
  SyncInsUpd := retval1;
end;

procedure TFBReplicator.SyncSrcAndTarget;
var
  retval1,retval2 : RETCODE;
  ErrorBuf : array[0..511] of Char;
  dbArray : HDBCARR;
  dbCount : Integer;
begin
  { For each record in the Change Table matching the passed in Loc_id.
    We'll Pull the information out of the Source DB and propagate it to the Target DB }
  FillChar(ErrorBuf, SizeOf(ErrorBuf), Char(0));
  dbArray[0] := @SrcHandle;
  dbArray[1] := @TargetHandle;
  dbCount := 2;
  //         SQLTransactMdb( hdbcArr : array of PHDBC;TxOption : UWORD;DbCount:Integer) : RETCODE;export;
  retval1 := SQLTransactMdb( dbArray, SQL_BEGIN_TRANS, dbCount);
  if ( retval1 = SQL_SUCCESS) then begin
    retval1 := OpenChanges;
    if (retval1 = SQL_SUCCESS) then begin
      if brVerbose then HandleError('Before GetChanges LOCID: '+IntToStr(tempLocId)+'; (SyncSrcAndTarget)', False);
      retval2 := GetChanges;
      retval1 := SQL_SUCCESS;
      if brVerbose then HandleError('Testing while loop; (SyncSrcAndTarget)', False);
      if brVerbose then HandleError('retval1='+IntToStr(retval1)+': retval2='+IntToStr(retval2)+'; (SyncSrcAndTarget)', False);
      if brVerbose then HandleError('ChgLocID: '+IntToStr(tempLocId)+'; (SyncSrcAndTarget)', False);
      while ( retval2 = SQL_SUCCESS) and (retval1 = SQL_SUCCESS) do begin
        {Get the proper SQL statement for the given TABLENAME, and Operation }
        SqlStmtPtr := SearchStmtLL(StmtHead,ChgTableName,ChgOperation);
        if brVerbose then HandleError('SQLStatement:'+String(SqlStmtPtr)+'; (SyncSrcAndTarget)', False);
        if SqlStmtPtr <> nil then begin
          { Handle the sync of Insert, Update, or Delete }
          if (ChgOperation = 'U') or (ChgOperation = 'I') then
            retval1 := SyncInsUpd
          else
            retval1 := SyncDelete;
          if brVerbose then HandleError('SyncInsUpd or SyncDelete called and finished; (SyncSrcAndTarget)', False);
          if (retval1 = SQL_SUCCESS) then begin
            {We've done what we should have, so we can delete the record from the Changes table.}
            retval1 := DeleteChangeRec;
            if brVerbose then HandleError('Deleted change record; (SyncSrcAndTarget)', False);
          end;
          retval2 := GetChanges;
          if brVerbose then HandleError('SQL_SUCCESS='+IntToStr(SQL_SUCCESS)+'; retval1='+IntToStr(retval1)+': retval2='+IntToStr(retval2)+'; (SyncSrcAndTarget)', False);
        end else begin
          HandleError('No Corresponding SQL Stmt for TableName='+ChgTableName+' Operation='+ChgOperation, True);
          SQLTransactMdb( dbArray,SQL_ROLLBACK,dbCount);
          // this may be a problem
//          ChgLocID := tempLocID;
          retval2 := 100;
        end;
      end;
      if (retval2 <> 100) or (retval1 <> SQL_SUCCESS) then begin
        if (retval2 <> 100) then begin
          HandleError('Sql Failed: '+ ErrorBuf, True);
          retval1 := SQLTransactMdb( dbArray,SQL_ROLLBACK,dbCount);
          // this may be a problem
//          ChgLocID := tempLocID;
        end
      end else
        retval1 := SQLTransactMdb( dbArray,SQL_COMMIT,dbCount);
      if brVerbose then HandleError('Closing Changes; (SyncSrcAndTarget)', False);
      if brVerbose then HandleError('ChgLocID: '+IntToStr(tempLocId)+'; (SyncSrcAndTarget)', False);
      if brVerbose then HandleError('tempLocID: '+IntToStr(tempLocId)+'; (SyncSrcAndTarget)', False);
      // this may be a problem
//      ChgLocID := tempLocID;
      CloseChanges;
    end else begin
      retval1 := SQLTransactMdb( dbArray,SQL_ROLLBACK,dbCount);
      // this may be a problem
//      ChgLocID := tempLocID;
    end
  end else begin
    SQLError(SrcHandle,nil,ErrorBuf);
    HandleError('Source Begin TX Failed: '+ ErrorBuf, True);
  end;
end;

procedure TFBReplicator.ReplicateData;
var
  PosCode,NumLocations : Integer;
  ret1,ret2 : RETCODE;
  Connected2Src : Boolean;
  LocPtr : LocsPtr;
begin
  ret1 := SQL_SUCCESS;
  Connected2Src := False;
  try
    NumLocations := CacheSqlStmts;
    if brVerbose then HandleError('SQL Statements cached; (ReplicateData)', False);
    { Cache all the SQL stmts for all tables that may be replicated. At the same time, figure out
      how many target locations there are and cache all of their DB names }
    if (DBSource.InTransaction) then
      DBSource.Commit;
    DBSource.StartTransaction;
    start := sysutils.now;
    if brShowStats then begin
      QChanges.Open;
      // across all sources and locations
      RowsToBeReplicated := QChangesCount.AsInteger;
      QChanges.Close;
    end;
    if brVerbose then HandleError('Source DB Transaction started; (ReplicateData)', False);
    QLocChanges.Open; { Query for distinct LocIds in Change table}
    if brVerbose then HandleError('QLocChanges open; (ReplicateData)', False);
    QLocChanges.First;
    if brVerbose then HandleError('QLocChanges at first record; (ReplicateData)', False);
    nLocations := 0;
    if brVerbose then HandleError('Distinct LOCIDs selected; (ReplicateData)', False);
    while ((QLocChanges.EOF <> True) and (ret1=SQL_SUCCESS)) do begin
      inc(nLocations);
      if brVerbose then HandleError('Processing LOCID: '+QLocChangesLOC_ID.AsString+'; (ReplicateData)', False);
      if (not Connected2Src) then begin
        ret1 := ConnectDb(-1,nil);  {-1 signifies the Source DB}
        ChgLocId := QLocChangesLOC_ID.value;
        tempLocID := ChgLocId;
        if brVerbose then HandleError('Assigned LOCID: '+IntToStr(tempLocId)+'; (ReplicateData)', False);
        if (ret1 = SQL_SUCCESS) then
          Connected2Src := True;
      end;
      ChgLocId := QLocChangesLOC_ID.value;
      if brVerbose then HandleError('Re-assigned LOCID: '+IntToStr(tempLocId)+'; (ReplicateData)', False);
      LocPtr := SearchLocsLL(LocsHead,ChgLocId);
      ret2 := ConnectDb(ChgLocId,LocPtr);
      if ((ret2 = SQL_SUCCESS) and (ret1 = SQL_SUCCESS)) then begin
        if brVerbose then HandleError('Connected to LOCID: '+IntToStr(tempLocId)+'; (ReplicateData)', False);
        SyncSrcAndTarget;
        if brVerbose then HandleError('SyncSrcandTarget returns LOCID: '+IntToStr(tempLocId)+'; (ReplicateData)', False);
        DisconnectDb(ChgLocId);
        if brVerbose then HandleError('Disconnected from LOCID: '+IntToStr(tempLocId)+'; (ReplicateData)', False);
      end;
      QLocChanges.Next;
    end;
    QLocChanges.Close;
    if brShowStats then begin
      QChanges.Open;
      RowsReplicated := RowsToBeReplicated - QChangesCount.AsInteger;
      QChanges.Close;
    end;
    finish := sysutils.now;
    DecodeTime((finish-start), Hour, Min, Sec, MSec);
    if brShowStats then begin
      HandleError(IntToStr(RowsReplicated)+' rows replicated from total of '+IntToStr(RowsToBeReplicated)+' rows, across '+IntToStr(nLocations)+' target locations for '+SrcPathDB, False);
      HandleError('Time taken to replicate: '+IntToStr(Hour)+' hrs, '+IntToStr(Min)+' mins, '+IntToStr(Sec)+' secs', False);
      if (RowsReplicated>0) then begin
        sEmailStats := sEmailStats +CRLF+'<p><font face="Tahoma" size="2">'+DateTimetoStr(Now)+': <b>'+IntToStr(RowsReplicated)+'</b> rows replicated from total of <b>'+IntToStr(RowsToBeReplicated)+'</b> rows, across <b>'+IntToStr(nLocations)+'</b> target locations for <b>'+SrcPathDB+'</b></font>';
        sEmailStats := sEmailStats +CRLF+'<br><font face="Tahoma" size="2">Time taken to replicate: '+IntToStr(Hour)+' hrs, '+IntToStr(Min)+' mins, '+IntToStr(Sec)+' secs</font></p>';
      end;
    end;
    DBSource.Commit;
    DBSource.Connected := False;
    if brVerbose then HandleError('Source DB disconnected; (ReplicateData)', False);
    if (Connected2Src) then
      DisconnectDb(-1);  { -1 signifies the Source DB }
  except
    HandleError('Exception in ReplicateData', True);
  end;
end;

procedure TFBReplicator.ServiceStart(Sender: TService;
  var Started: Boolean);
begin
  Read_Registry;
  BlobMaxSize := irMaxBlobSize;
  TReplInterval.Interval := StrToInt(srInterval)*1000*60;
  if TReplInterval.Interval < 1 then
     TReplInterval.Enabled := False
  else
    TReplInterval.Enabled := True;
  if brEvtStr then
    Reg4Event := True
  else
    Reg4Event := False;
  TDoReplication.Enabled := True;
  if brVerbose then begin
    HandleError('---------- Service Started ----------', False);
    HandleError('--------- Verbose Logging ON --------', False);
  end else begin
    HandleError('---------- Service Started ----------', False);
    HandleError('-------- Verbose Logging OFF --------', False);
  end;
  // Send Start email
  SendStartingEmail;
end;

procedure TFBReplicator.IBEvent1EventAlert(Sender: TObject; EventName: string;
  EventCount: Longint; var CancelAlerts: Boolean);
begin
  TDoReplication.Enabled := True;
  if brVerbose then HandleError('DoReplication enabled by event alert; (IBEvent1EventAlert)', False);
end;

procedure ServiceController(CtrlCode: DWord); stdcall;
begin
  FBReplicator.Controller(CtrlCode);
end;

function TFBReplicator.GetServiceController: TServiceController;
begin
  Result := ServiceController;
end;

procedure TFBReplicator.TDoReplicationTimer(Sender: TObject);
var
  nTotalRowsToBeReplicated: Integer;
begin
  SetParams;
  // ReplDB is now connected (or fatalerror is True)
  if (ReplDB.InTransaction) then
    ReplDB.Commit;
  ReplDB.StartTransaction;
  QSrcLocs.Open;
  if brVerbose then HandleError('Source query open; (TDoReplicationTimer)', False);
  QSrcLocs.First;
  // Replicate all source DBs registered
  while not QSrcLocs.Eof do begin
    inc(nSourceCount);
    // Maybe needs additional threads for multiple sources.
    // But OK if one or more sources
    // Repl_tables entries must still rely on the table names across all
    // of the multiple sources being unique
    // OR, the multiple sources share the same structure
    // and the replicator instance can be used to do both way replication
    DBSource.Server := QSrcLocsSOURCE_SERVER.value;
    DBSource.Path := QSrcLocsSOURCE_PATH.value;
    DBSource.Username := QSrcLocsREPLUSER.value;
    DBSource.Password := QSrcLocsREPLPASSWD.value;
    SrcPathDB := QSrcLocsSOURCE_SERVER.value+':'+QSrcLocsSOURCE_PATH.value;
    SrcUsername := QSrcLocsREPLUSER.value;
    SrcPassword := QSrcLocsREPLPASSWD.value;
{    EvSourceDB.Server := QSrcLocsSOURCE_SERVER.value;
    EvSourceDB.Path := QSrcLocsSOURCE_PATH.value;
    EvSourceDB.Username := QSrcLocsREPLPASSWD.value;
    EvSourceDB.Password := QSrcLocsREPLPASSWD.value;}
    if brVerbose then HandleError('Replicating source: '+SrcPathDB+'; (TDoReplicationTimer)', False);
    try
      if not (FatalError) then begin
        if not (DBSource.Connected) then begin
          DBSource.Connected := True;
          if brVerbose then HandleError('Source DB open; (TDoReplicationTimer)', False);
        end;
{        if not (EvSourceDB.Connected) then
          EvSourceDB.Connected := True;
        if (Reg4Event) then
          IBEvent.Registered := True;}
      end else
        HandleError('Fatal Error', True);
    except
      HandleError('Error connecting to source or target database; (TDoReplicationTimer)', True);
    end;
    TDoReplication.Enabled := False;
    if brVerbose then HandleError('DoReplication disabled; (TDoReplicationTimer)', False);
    if ( Not Running) then begin
      Running := True;
      if brVerbose then HandleError('Calling ReplicateData; (TDoReplicationTimer)', False);
      ReplicateData;
      nTotalRowsReplicated := nTotalRowsReplicated + RowsReplicated;
      if brVerbose then HandleError('ReplicateData complete; (TDoReplicationTimer)', False);
      if (AnyErrors > 0) then begin
        // Send warning email
        if brUseEmailAlerts and (TotalErrors>irTolerance) then begin
          IdSMTP1.Host := srSMTPHost;
          if brUseSMTPAuthentication then begin
            IdSMTP1.AuthType := atDefault;
            IdSMTP1.Username := srSMTPLogin;
            IdSMTP1.Password := srSMTPPWord;
          end else begin
            IdSMTP1.AuthType := atNone;
            IdSMTP1.Username := '';
            IdSMTP1.Password := '';
          end;
          try
            IdSMTP1.Connect;
          except
            on E:Exception do begin
              HandleError('Error connecting to SMTP server: '+E.Message, False);
            end;
          end;
          // Send warning email
          idMessage1.From.Name := srReplServer;
          idMessage1.From.Address := srReplServer+'@'+srFromAddress;
          idMessage1.Sender.Address := srReplServer+'@'+srFromAddress;
          idMessage1.Sender.Name := srReplServer;
          idMessage1.Recipients[0].Name := '';
          idMessage1.Recipients[0].Address := srEmail;
          idMessage1.ReplyTo[0].Name := srReplServer;
          idMessage1.ReplyTo[0].Address := srReplServer+'@'+srFromAddress;
          idMessage1.Subject := 'FBReplicator Error Message';
          idMessage1.Body.Clear;
          idMessage1.Body.Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2//EN">');
          idMessage1.Body.Add('<html>');
          idMessage1.Body.Add('<head>');
          idMessage1.Body.Add('<META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=iso-8859-1">');
          idMessage1.Body.Add('<title>FBReplicator Emailer</title>');
          idMessage1.Body.Add('</head>');
          idMessage1.Body.Add('<body>');
          idMessage1.Body.Add('<table width="100%" cellspacing=0 cellpadding=3 border=0>');
          idMessage1.Body.Add('<TR>');
          idMessage1.Body.Add('<TD width=32 background=http://www.meta.com.au/images/stories/fbheader_tile.gif>');
          idMessage1.Body.Add('<IMG alt=meta src="http://www.meta.com.au/images/stories/fbrepllogo.gif"></TD>');
          idMessage1.Body.Add('<TD background=http://www.meta.com.au/images/stories/fbheader_tile.gif><font face="Tahoma" size="2"><b>FBReplicator</b></font><br><font face="Tahoma" size="1">This email has been server generated, please do not reply.</font></TD>');
          idMessage1.Body.Add('</TR></table>');
          idMessage1.Body.Add('<p><font face="Tahoma" size="2">There has been an error during replication on '+srReplServer+'<br>while trying to replicate source database:<br>'+DateTimeToStr(Now)+': <b>'+ SrcPathDB+'.</b></font></p>');
          idMessage1.Body.Add('<p><font face="Tahoma" size="2">There has been <b>'+IntToStr(TotalErrors)+'</b> error(s) during this session which exceeds the set tolerance of <b>'+IntToStr(irTolerance)+'</b> errors. The error count has been reset.</font></p>');
          idMessage1.Body.Add('<p><font face="Tahoma" size="2">The latest error message was:<br><b>'+LatestError+'</b></font></p>');
          if brShowStats then begin
            idMessage1.Body.Add('<p><font face="Tahoma" size="2"><b>'+IntToStr(RowsToBeReplicated)+'</b> rows still to replicate across '+IntToStr(nLocations)+' target location(s)</font></p>');
          end;
          idMessage1.Body.Add('<p><font face="Tahoma" size="1">This email has been sent by the FBReplicator Server to warn of errors currently occuring at the server.');
          idMessage1.Body.Add('<br><b>Warning</b>: This computer program is produced as shareware. ');
          idMessage1.Body.Add('The source can be copied, freely distributed, modified, sold or anything else that you wish to do with it. ');
          idMessage1.Body.Add('Certain parts, however, are copyright and it is recommended that appropriate recognition of the copyright holders be displayed somewhere in the distribution.</font>');
          idMessage1.Body.Add('</body>');
          idMessage1.Body.Add('</html>');
          try
            IdSMTP1.Send(idMessage1);
          except
            on E:Exception do begin
              HandleError('Error sending error email message: '+E.Message, False);
            end;
          end;
          IdSMTP1.Disconnect;
          if brVerbose then HandleError('Total errors of: '+IntToStr(TotalErrors)+', exceeds tolerance of: '+IntToStr(irTolerance), False);
          TotalErrors := 0;
          SuccessCycles := 0;
          RowsReplicated := 0;
          nLocations := 0;
        end;
      end else begin
        if brVerbose then HandleError('Source replication complete. Next cycle in '+IntToStr(Trunc(TReplInterval.Interval/1000/60))+' minutes; (TDoReplicationTimer)', False);
      end;
      AnyErrors := 0;
      Running := False;
    end;
    QSrcLocs.Next;
    if not (FatalError) then begin
      if (DBSource.Connected) then
        DBSource.Connected := False;
{      if (Reg4Event) then
        IBEvent.Registered := False;}
{      if (EvSourceDB.Connected) then
        EvSourceDB.Connected := False;}
    end else
      HandleError('Fatal Error disconnecting', True);
  end;
  // END Replicate all source DBs registered
  HandleError('Total rows replicated this cycle: '+IntToStr(nTotalRowsReplicated), False);
  if SuccessCycles>=irHeartBeat then begin
    if brVerbose then HandleError('Successful Cycles: '+IntToStr(SuccessCycles)+'; Hearbeat: '+IntToStr(irHeartBeat)+'; (TDoReplicationTimer)', False);
    if breHeartbeat then begin  
      //**************
      // Send success email
      //**************
      IdSMTP1.Host := srSMTPHost;
      if brUseSMTPAuthentication then begin
        IdSMTP1.Username := srSMTPLogin;
        IdSMTP1.Password := srSMTPPWord;
        IdSMTP1.AuthType := atDefault;
      end else begin
        IdSMTP1.Username := '';
        IdSMTP1.Password := '';
        IdSMTP1.AuthType := atNone;
      end;
      try
        IdSMTP1.Connect;
      except
        on E:Exception do begin
          HandleError('Error connecting to SMTP server: '+E.Message, False);
        end;
      end;
      // Send heartbeat email
      idMessage1.From.Name := srReplServer;
      idMessage1.From.Address := srReplServer+'@'+srFromAddress;
      idMessage1.Sender.Address := srReplServer+'@'+srFromAddress;
      idMessage1.Sender.Name := srReplServer;
      idMessage1.Recipients[0].Name := '';
      idMessage1.Recipients[0].Address := srEmail;
      idMessage1.ReplyTo[0].Name := srReplServer;
      idMessage1.ReplyTo[0].Address := srReplServer+'@'+srFromAddress;
      idMessage1.Subject := 'FBReplicator Hearbeat Message';
      idMessage1.Body.Clear;
      idMessage1.Body.Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2//EN">');
      idMessage1.Body.Add('<html>');
      idMessage1.Body.Add('<head>');
      idMessage1.Body.Add('<META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=iso-8859-1">');
      idMessage1.Body.Add('<title>FBReplicator Emailer</title>');
      idMessage1.Body.Add('</head>');
      idMessage1.Body.Add('<body>');
      idMessage1.Body.Add('<table width="100%" cellspacing=0 cellpadding=3 border=0>');
      idMessage1.Body.Add('<TR>');
      idMessage1.Body.Add('<TD width=32 background=http://www.meta.com.au/images/stories/fbheader_tile.gif>');
      idMessage1.Body.Add('<IMG alt=meta src="http://www.meta.com.au/images/stories/fbrepllogo.gif"></TD>');
      idMessage1.Body.Add('<TD background=http://www.meta.com.au/images/stories/fbheader_tile.gif><font face="Tahoma" size="2"><b>FBReplicator</b></font><br><font face="Tahoma" size="1">This email has been server generated, please do not reply.</font></TD>');
      idMessage1.Body.Add('</TR></table>');
      idMessage1.Body.Add('<p><font face="Tahoma" size="2">'+DateTimetoStr(SysUtils.Now)+': Heartbeat from: <b>'+srReplServer+'</b> after '+IntToStr(SuccessCycles)+' successful cycles. Success cycle count and replicated row has been reset.</font></p>');
      if brShowStats then begin
        idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>ON</b>.</font></p>');
        idMessage1.Body.Add(sEmailStats);
      end else
        idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>OFF</b>.</font></p>');
      idMessage1.Body.Add('<p><font face="Tahoma" size="1">This email has been sent by the FBReplicator Server to confirm continuing operation of the service.');
      idMessage1.Body.Add('<br><b>Warning</b>: This computer program is produced as shareware. ');
      idMessage1.Body.Add('The source can be copied, freely distributed, modified, sold or anything else that you wish to do with it. ');
      idMessage1.Body.Add('Certain parts, however, are copyright and it is recommended that appropriate recognition of the copyright holders be displayed somewhere in the distribution.</font>');
      idMessage1.Body.Add('</body>');
      idMessage1.Body.Add('</html>');
      try
        IdSMTP1.Send(idMessage1);
      except
        on E:Exception do begin
          HandleError('Error sending error email message: '+E.Message, False);
        end;
      end;
      IdSMTP1.Disconnect;
      HandleError('Heartbeat email sent', False);
      SuccessCycles := 0;
      RowsReplicated := 0;
      nLocations := 0;
      TotalErrors := 0;
      sEmailStats := '';
    end;
  end;
  nSourceCount := 0;
  QSrcLocs.Close;
  HandleError('********** Cycle Complete ***********', False);
  Inc(SuccessCycles);
  nTotalRowsReplicated := 0;
  ReplDB.Commit;
  ReplDB.Connected := False;
end;


procedure TFBReplicator.TReplIntervalTimer(Sender: TObject);
begin
  TDoReplication.Enabled := True;
  if brVerbose then HandleError('DoReplication enabled by ReplIntervalTimer; (TReplIntervalTimer)', False);
end;

procedure TFBReplicator.ServiceStop(Sender: TService;
  var Stopped: Boolean);
begin
  Read_Registry;
  BlobMaxSize := irMaxBlobSize;
  FreeSqlStmts; { Free all Cached SQL stmts }
  if ( DBSource.Connected) then
   DBSource.Connected := False;
{  if ( EvSourceDB.Connected) then
  begin
   IBEvent.Registered := False;
   EvSourceDB.Connected := False;
  end;}
  ReplDB.Connected := False;
  if brVerbose then begin
    HandleError('---------- Service Stopped ----------', False);
    HandleError('--------- Verbose Logging ON --------', False);
  end else begin
    HandleError('---------- Service Stopped ----------', False);
    HandleError('-------- Verbose Logging OFF --------', False);
  end;
  SendStoppingEmail;
end;

procedure TFBReplicator.SendStartingEmail;
begin
  //**************
  // Send success email
  //**************
  IdSMTP1.Host := srSMTPHost;
  if brUseSMTPAuthentication then begin
    IdSMTP1.Username := srSMTPLogin;
    IdSMTP1.Password := srSMTPPWord;
    IdSMTP1.AuthType := atDefault;
  end else begin
    IdSMTP1.Username := '';
    IdSMTP1.Password := '';
    IdSMTP1.AuthType := atNone;
  end;
  try
    IdSMTP1.Connect;
  except
    on E:Exception do begin
      HandleError('Error connecting to SMTP server: '+E.Message, False);
    end;
  end;
  // Send heartbeat email
  idMessage1.From.Name := srReplServer;
  idMessage1.From.Address := srReplServer+'@'+srFromAddress;
  idMessage1.Sender.Address := srReplServer+'@'+srFromAddress;
  idMessage1.Sender.Name := srReplServer;
  idMessage1.Recipients[0].Name := '';
  idMessage1.Recipients[0].Address := srEmail;
  idMessage1.ReplyTo[0].Name := srReplServer;
  idMessage1.ReplyTo[0].Address := srReplServer+'@'+srFromAddress;
  idMessage1.Subject := 'FBReplicator Service Start Message';
  idMessage1.Body.Clear;
  idMessage1.Body.Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2//EN">');
  idMessage1.Body.Add('<html>');
  idMessage1.Body.Add('<head>');
  idMessage1.Body.Add('<META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=iso-8859-1">');
  idMessage1.Body.Add('<title>FBReplicator Emailer</title>');
  idMessage1.Body.Add('</head>');
  idMessage1.Body.Add('<body>');
  idMessage1.Body.Add('<table width="100%" cellspacing=0 cellpadding=3 border=0>');
  idMessage1.Body.Add('<TR>');
  idMessage1.Body.Add('<TD width=32 background=http://www.meta.com.au/images/stories/fbheader_tile.gif>');
  idMessage1.Body.Add('<IMG alt=meta src="http://www.meta.com.au/images/stories/fbrepllogo.gif"></TD>');
  idMessage1.Body.Add('<TD background=http://www.meta.com.au/images/stories/fbheader_tile.gif><font face="Tahoma" size="2"><b>FBReplicator</b></font><br><font face="Tahoma" size="1">This email has been server generated, please do not reply.</font></TD>');
  idMessage1.Body.Add('</TR></table>');
  idMessage1.Body.Add('<p><font face="Tahoma" size="2">'+DateTimeToStr(Now)+': Service started on: <b>'+srReplServer+'</b>.</font></p>');
  if brShowStats then begin
    idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>ON</b>.</font></p>');
    idMessage1.Body.Add(sEmailStats);
  end else
    idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>OFF</b>.</font></p>');
  idMessage1.Body.Add('<p><font face="Tahoma" size="1">This email has been sent by the FBReplicator Server to confirm continuing operation of the service.');
  idMessage1.Body.Add('<br><b>Warning</b>: This computer program is produced as shareware. ');
  idMessage1.Body.Add('The source can be copied, freely distributed, modified, sold or anything else that you wish to do with it. ');
  idMessage1.Body.Add('Certain parts, however, are copyright and it is recommended that appropriate recognition of the copyright holders be displayed somewhere in the distribution.</font>');
  idMessage1.Body.Add('</body>');
  idMessage1.Body.Add('</html>');
  try
    IdSMTP1.Send(idMessage1);
  except
    on E:Exception do begin
      HandleError('Error sending service start email message: '+E.Message, False);
    end;
  end;
  IdSMTP1.Disconnect;
  HandleError('Service Starting email sent', False);
end;

procedure TFBReplicator.SendStoppingEmail;
begin
  //**************
  // Send success email
  //**************
  IdSMTP1.Host := srSMTPHost;
  if brUseSMTPAuthentication then begin
    IdSMTP1.Username := srSMTPLogin;
    IdSMTP1.Password := srSMTPPWord;
    IdSMTP1.AuthType := atDefault;
  end else begin
    IdSMTP1.Username := '';
    IdSMTP1.Password := '';
    IdSMTP1.AuthType := atNone;
  end;
  try
    IdSMTP1.Connect;
  except
    on E:Exception do begin
      HandleError('Error connecting to SMTP server: '+E.Message, False);
    end;
  end;
  // Send heartbeat email
  idMessage1.From.Name := srReplServer;
  idMessage1.From.Address := srReplServer+'@'+srFromAddress;
  idMessage1.Sender.Address := srReplServer+'@'+srFromAddress;
  idMessage1.Sender.Name := srReplServer;
  idMessage1.Recipients[0].Name := '';
  idMessage1.Recipients[0].Address := srEmail;
  idMessage1.ReplyTo[0].Name := srReplServer;
  idMessage1.ReplyTo[0].Address := srReplServer+'@'+srFromAddress;
  idMessage1.Subject := 'FBReplicator Service Stop Message';
  idMessage1.Body.Clear;
  idMessage1.Body.Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2//EN">');
  idMessage1.Body.Add('<html>');
  idMessage1.Body.Add('<head>');
  idMessage1.Body.Add('<META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=iso-8859-1">');
  idMessage1.Body.Add('<title>FBReplicator Emailer</title>');
  idMessage1.Body.Add('</head>');
  idMessage1.Body.Add('<body>');
  idMessage1.Body.Add('<table width="100%" cellspacing=0 cellpadding=3 border=0>');
  idMessage1.Body.Add('<TR>');
  idMessage1.Body.Add('<TD width=32 background=http://www.meta.com.au/images/stories/fbheader_tile.gif>');
  idMessage1.Body.Add('<IMG alt=meta src="http://www.meta.com.au/images/stories/fbrepllogo.gif"></TD>');
  idMessage1.Body.Add('<TD background=http://www.meta.com.au/images/stories/fbheader_tile.gif><font face="Tahoma" size="2"><b>FBReplicator</b></font><br><font face="Tahoma" size="1">This email has been server generated, please do not reply.</font></TD>');
  idMessage1.Body.Add('</TR></table>');
  idMessage1.Body.Add('<p><font face="Tahoma" size="2">'+DateTimeToStr(Now)+': Service stopping on: <b>'+srReplServer+'</b>.</font></p>');
  if brShowStats then begin
    idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>ON</b>.</font></p>');
    idMessage1.Body.Add(sEmailStats);
  end else
    idMessage1.Body.Add('<p><font face="Tahoma" size="2">Statistics are currently <b>OFF</b>.</font></p>');
  idMessage1.Body.Add('<p><font face="Tahoma" size="1">This email has been sent by the FBReplicator Server to confirm continuing operation of the service.');
  idMessage1.Body.Add('<br><b>Warning</b>: This computer program is produced as shareware. ');
  idMessage1.Body.Add('The source can be copied, freely distributed, modified, sold or anything else that you wish to do with it. ');
  idMessage1.Body.Add('Certain parts, however, are copyright and it is recommended that appropriate recognition of the copyright holders be displayed somewhere in the distribution.</font>');
  idMessage1.Body.Add('</body>');
  idMessage1.Body.Add('</html>');
  try
    IdSMTP1.Send(idMessage1);
  except
    on E:Exception do begin
      HandleError('Error sending service stop email message: '+E.Message, False);
    end;
  end;
  IdSMTP1.Disconnect;
  HandleError('Service Stopping email sent', False);
end;

initialization
  SomeNull1 := False;
  SomeNull2 := False;
  SomeNull3 := False;
  SomeNull4 := False;
  SomeNull5 := False;
  AnyErrors := 0;
  TotalErrors := 0;
  SuccessCycles := 0;
  RowsReplicated := 0;
  RowsToBeReplicated := 0;
  nLocations := 0;
  FatalError := False;
  Running := False;
  CachedEm := False;
  nTotalRowsReplicated := 0;
  sEmailStats := '';
end.
